class VersionSelector {
    constructor(instanceElement) {
        this.container = instanceElement;
        // Use global variable from template (Hugo's absURL) with fallback
        this.versionsUrl = window.versionsJsonUrl || '/versions.json';
        this.maxRecentVersions = 3;
        // Use global variable from template (supervisor's suggestion)
        this.currentVersion = window.actualVersion || this.detectCurrentVersion();

        this.currentVersionEl = this.container.querySelector('.js-current-version-display');
        this.menu = this.container.querySelector('.js-version-dropdown-menu');
        // this.toggleButton = this.container.querySelector('.js-version-selector-toggle'); // Added for completeness, if needed later

        this.init();
    }

    async init() {
        if (!this.currentVersionEl || !this.menu) {
            console.warn("Essential elements not found in this version selector instance:", this.container);
            return; // Stop initialization for this specific instance
        }
        try {
            const versions = await this.fetchVersions();
            this.renderVersionSelector(versions);
        } catch (error) {
            console.warn('Failed to load dynamic versions, falling back to static:', error);
            this.showLoadingError();
            // Fallback to existing Hugo-generated selector - no additional action needed
        }
    }

    async fetchVersions() {
        const response = await fetch(this.versionsUrl);
        if (!response.ok) throw new Error(`HTTP ${response.status}`);
        return await response.json();
    }

    detectCurrentVersion() {
        // Fallback detection (only used if global variable not available)
        const path = window.location.pathname;
        if (path.includes('/cloud/')) {
            return 'Cloud';
        }
        const versionMatch = path.match(/\/cloud-native\/([\d.]+)\//);
        return versionMatch ? `v${versionMatch[1]}` : 'Cloud';
    }

    findCurrentVersionData(allVersions) {
        // Simple direct matching using currentVersion string
        return allVersions.find(v => v.version === this.currentVersion);
    }

    renderVersionSelector(allVersions) {
        
        if (!this.currentVersionEl || !this.menu) return;

        // Find current version data
        const currentVersionData = this.findCurrentVersionData(allVersions);
        
        // Update current version display
        if (currentVersionData) {
            this.currentVersionEl.textContent = `Version ${currentVersionData.version}`;
        } else if (this.currentVersion) {
            // Fallback to detected version
            this.currentVersionEl.textContent = `Version ${this.currentVersion}`;
        }

        // Clear existing menu and rebuild
        this.menu.textContent = '';

        // Add recent versions
        const recentVersions = allVersions.slice(0, this.maxRecentVersions);
        recentVersions.forEach(version => {
            const isActive = currentVersionData && version.version === currentVersionData.version;
            const item = this.createVersionItem(version, isActive);
            this.menu.appendChild(item);
        });

        // Add separator and archive link if there are more versions
        if (allVersions.length > this.maxRecentVersions) {
            this.menu.appendChild(this.createSeparator());
            this.menu.appendChild(this.createArchiveLink());
        }
    }

    createVersionItem(version, isActive = false) {
        const link = document.createElement('a');
        link.className = `dropdown-item${isActive ? ' active' : ''}`;
        
        // Use baseUrl from Hugo template to construct proper URLs for all environments
        if (window.baseUrl && version.url.startsWith('/')) {
            // Remove leading slash from version.url and combine with baseUrl
            link.href = window.baseUrl + version.url.substring(1);
        } else {
            // Fallback to original behavior
            link.href = version.url;
        }
        
        link.textContent = version.version;
        
        // Add aria-current for screen readers to indicate current page
        if (isActive) {
            link.setAttribute('aria-current', 'page');
        }
        
        return link;
    }

    createSeparator() {
        const separator = document.createElement('div');
        separator.className = 'dropdown-divider';
        return separator;
    }

    createArchiveLink() {
        const link = document.createElement('a');
        link.className = 'dropdown-item';
        
        const currentHostname = window.location.hostname;
        if (currentHostname.includes('netlify.app')) {
            // For Netlify previews, link to the versions page within the current preview site
            link.href = window.location.origin + '/docs/versions/';
        } else if (currentHostname === 'localhost' || currentHostname === '127.0.0.1') {
            // For local testing, link to the versions page
            link.href = '/versions/';
        } else {
            // For production - link to the unversioned archive page
            link.href = 'https://www.gooddata.com/docs/versions/';
        }
        
        // Safe DOM manipulation - XSS protected
        const icon = document.createElement('i');
        icon.className = 'fas fa-archive me-2';
        icon.setAttribute('aria-hidden', 'true');
        
        link.appendChild(icon);
        link.appendChild(document.createTextNode('Archive'));
        
        // Add descriptive text for screen readers
        link.setAttribute('aria-label', 'View all versions in archive');
        
        return link;
    }

    showLoadingError() {
        if (this.menu) {
            // Only show error if menu is currently showing loading spinner
            const loadingSpinner = this.menu.querySelector('.spinner-border');
            if (loadingSpinner) {
                // Clear existing content safely
                this.menu.textContent = '';
                
                // Create error message with proper DOM methods
                const errorDiv = document.createElement('div');
                errorDiv.className = 'dropdown-item-text text-muted small';
                errorDiv.setAttribute('role', 'alert');
                errorDiv.setAttribute('aria-live', 'polite');
                
                const icon = document.createElement('i');
                icon.className = 'fas fa-exclamation-triangle me-1';
                icon.setAttribute('aria-hidden', 'true');
                
                errorDiv.appendChild(icon);
                errorDiv.appendChild(document.createTextNode('Unable to load latest versions'));
                this.menu.appendChild(errorDiv);
            }
        }
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    // Only initialize if we have a version selector

    const selectorInstances = document.querySelectorAll(".js-version-selector-instance");
    selectorInstances.forEach(instanceElement => {
        new VersionSelector(instanceElement);
    });

}); 