// Prism.js error fix for "Cannot read properties of undefined (reading 'normalize')"
// This script patches the normalize-whitespace plugin to handle race conditions

(function() {
    'use strict';
    
    // Only run if Prism exists
    if (typeof window.Prism === 'undefined') {
        return;
    }
    
    // Wait for Prism to be fully loaded and initialized
    function initPrismFix() {
        // Check if NormalizeWhitespace plugin exists and has the normalize method
        if (window.Prism.plugins && 
            window.Prism.plugins.NormalizeWhitespace && 
            typeof window.Prism.plugins.NormalizeWhitespace.normalize === 'function') {
            return;
        }
        
        // Create a safe fallback for the normalize method
        if (window.Prism.plugins && window.Prism.plugins.NormalizeWhitespace) {
            var plugin = window.Prism.plugins.NormalizeWhitespace;
            
            // If normalize method is missing, create a fallback
            if (typeof plugin.normalize !== 'function') {
                plugin.normalize = function(code, settings) {
                    // Simple fallback that just returns the code as-is
                    return code || '';
                };
            }
        } else {
            // Create the entire plugin structure if missing
            if (!window.Prism.plugins) {
                window.Prism.plugins = {};
            }
            
            window.Prism.plugins.NormalizeWhitespace = {
                normalize: function(code, settings) {
                    // Simple fallback that just returns the code as-is
                    return code || '';
                },
                setDefaults: function(defaults) {
                    // No-op fallback
                },
                // Add other common methods that might be expected
                defaults: {
                    'remove-trailing': true,
                    'remove-indent': true,
                    'left-trim': true,
                    'right-trim': true
                }
            };
        }
    }
    
    // Try to initialize immediately if Prism is ready
    if (window.Prism && window.Prism.hooks) {
        initPrismFix();
    } else {
        // Wait for Prism to load
        var checkPrism = function() {
            if (window.Prism && window.Prism.hooks) {
                initPrismFix();
            } else {
                setTimeout(checkPrism, 50);
            }
        };
        
        // Start checking after a brief delay
        setTimeout(checkPrism, 100);
    }
    
    // Also patch any existing hooks that might call normalize
    if (window.Prism && window.Prism.hooks && window.Prism.hooks.add) {
        // Override the before-sanity-check hook to be safer
        var originalHooks = [];
        
        // Store original hooks if they exist
        if (window.Prism.hooks.all && window.Prism.hooks.all['before-sanity-check']) {
            originalHooks = window.Prism.hooks.all['before-sanity-check'].slice();
        }
        
        // Clear the hooks array and add a safe wrapper
        if (window.Prism.hooks.all) {
            window.Prism.hooks.all['before-sanity-check'] = [];
        }
        
        // Add our safe wrapper hook
        window.Prism.hooks.add('before-sanity-check', function(env) {
            try {
                // Ensure the normalize plugin is available before running original hooks
                initPrismFix();
                
                // Run original hooks safely
                for (var i = 0; i < originalHooks.length; i++) {
                    try {
                        originalHooks[i](env);
                    } catch (e) {
                        console.warn('Prism hook error (safely handled):', e);
                    }
                }
            } catch (e) {
                console.warn('Prism before-sanity-check hook error (safely handled):', e);
            }
        });
    }
})();